--Vorgaben für mögliche Fehlertypen

/*Fehlertypen in Fremdsprachen */
CREATE TABLE qabfehler(
  qfeh_id               SERIAL          PRIMARY KEY,
  qfeh_bez              VARCHAR(80)     NOT NULL,
  qfeh_grup             VARCHAR(80),
  qfeh_isService        BOOLEAN         DEFAULT FALSE,
  qfeh_inaktiv          BOOLEAN NOT NULL DEFAULT FALSE
);

CREATE OR REPLACE FUNCTION qabfehler__a_iud() RETURNS TRIGGER AS $$ -- #7018
  BEGIN
    IF NOT tg_op='INSERT' THEN
        DELETE FROM qabfehlertxt WHERE qfehl_qfeh_id=old.qfeh_id AND qfehl_spr_key=prodat_languages.current_lang();
    END IF;
    IF NOT tg_op='DELETE' THEN
        INSERT INTO qabfehlertxt (qfehl_qfeh_id, qfehl_spr_key, qfehl_txt) VALUES (new.qfeh_id, prodat_languages.current_lang(), new.qfeh_bez);
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER qabfehler__a_iud
  AFTER INSERT OR UPDATE OR DELETE
  ON qabfehler
  FOR EACH ROW
  EXECUTE PROCEDURE qabfehler__a_iud();

CREATE TABLE qabfehlertxt                                             -- #7018
 (qfehl_id               SERIAL NOT NULL PRIMARY KEY,
  qfehl_qfeh_id          INTEGER NOT NULL REFERENCES qabfehler ON DELETE CASCADE,
  qfehl_spr_key          VARCHAR(5) NOT NULL REFERENCES sprach,
  qfehl_txt              VARCHAR(80)
 );

CREATE OR REPLACE FUNCTION lang_qabfehler(INTEGER, VARCHAR) RETURNS VARCHAR AS $$    -- #7018
DECLARE result VARCHAR;
  BEGIN
   SELECT COALESCE(qfehl_txt,qfeh_bez) INTO result
    FROM qabfehler
      LEFT JOIN qabfehlertxt ON qfehl_qfeh_id = qfeh_id AND qfehl_spr_key = $2
   WHERE qfeh_id=$1;
  RETURN result;
END $$ LANGUAGE plpgsql;

/*qabentscheid-Table in Fremdsprachen */
CREATE TABLE qabentscheid(
  qent_id               SERIAL          PRIMARY KEY,
  qent_bez              VARCHAR(80)     NOT NULL,
  qent_isService        BOOLEAN         DEFAULT FALSE,
  qent_inaktiv          BOOLEAN NOT NULL DEFAULT FALSE
);

CREATE OR REPLACE FUNCTION qabentscheid__a_iud() RETURNS TRIGGER AS $$   -- #7018
  BEGIN
    IF NOT tg_op='INSERT' THEN
        DELETE FROM qabentscheidtxt WHERE qentl_qent_id=old.qent_id AND qentl_spr_key=prodat_languages.current_lang();
    END IF;
    IF NOT tg_op='DELETE' THEN
        INSERT INTO qabentscheidtxt (qentl_qent_id, qentl_spr_key, qentl_txt) VALUES (new.qent_id, prodat_languages.current_lang(), new.qent_bez);
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER qabentscheid__a_iud
  AFTER INSERT OR UPDATE OR DELETE
  ON qabentscheid
  FOR EACH ROW
  EXECUTE PROCEDURE qabentscheid__a_iud();

CREATE TABLE qabentscheidtxt                                             -- #7018
 (qentl_id               SERIAL NOT NULL PRIMARY KEY,
  qentl_qent_id          INTEGER NOT NULL REFERENCES qabentscheid ON DELETE CASCADE,
  qentl_spr_key          VARCHAR(5) NOT NULL REFERENCES sprach,
  qentl_txt              VARCHAR(80)
 );

CREATE OR REPLACE FUNCTION lang_qabentscheid(INTEGER, VARCHAR) RETURNS VARCHAR AS $$         -- #7018
DECLARE result VARCHAR;
  BEGIN
    SELECT COALESCE(qentl_txt, qent_bez) INTO result
     FROM qabentscheid
       LEFT JOIN qabentscheidtxt ON qentl_qent_id=qent_id AND qentl_spr_key=$2
    WHERE qent_id=$1;
  RETURN result;
END $$ LANGUAGE plpgsql;

/*qabklasse-Table in Fremdsprachen */
CREATE TABLE qabklasse(
  qkla_id               SERIAL          PRIMARY KEY,
  qkla_bez              VARCHAR(80)     NOT NULL,
  qkla_isService        BOOLEAN         DEFAULT FALSE,
  qkla_inaktiv          BOOLEAN NOT NULL DEFAULT FALSE
);

CREATE OR REPLACE FUNCTION qabklasse__a_iud() RETURNS TRIGGER AS $$                  -- #7018
  BEGIN
    IF NOT tg_op='INSERT' THEN
        DELETE FROM qabklassetxt WHERE qklal_qkla_id=old.qkla_id AND qklal_spr_key=prodat_languages.current_lang();
    END IF;
    IF NOT tg_op='DELETE' THEN
        INSERT INTO qabklassetxt (qklal_qkla_id, qklal_spr_key, qklal_txt) VALUES (new.qkla_id, prodat_languages.current_lang(), new.qkla_bez);
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER qabklasse__a_iud                                                    -- #7018
  AFTER INSERT OR UPDATE OR DELETE
  ON qabklasse
  FOR EACH ROW
  EXECUTE PROCEDURE qabklasse__a_iud();

CREATE TABLE qabklassetxt                                                            -- #7018
 (qklal_id               SERIAL NOT NULL PRIMARY KEY,
  qklal_qkla_id          INTEGER NOT NULL REFERENCES qabklasse ON DELETE CASCADE,
  qklal_spr_key          VARCHAR(5) NOT NULL REFERENCES sprach,
  qklal_txt              VARCHAR(80)
 );

CREATE OR REPLACE FUNCTION lang_qabklasse(INTEGER, VARCHAR) RETURNS VARCHAR AS $$    -- #7018
DECLARE result VARCHAR;
  BEGIN
   SELECT COALESCE(qklal_txt, qkla_bez) INTO result
    FROM qabklasse
      LEFT JOIN qabklassetxt ON qklal_qkla_id=qkla_id AND qklal_spr_key=$2
   WHERE qkla_id = $1;
  RETURN result;
END $$ LANGUAGE plpgsql;

/*qababteilung-Table in Fremdsprachen */
CREATE TABLE qababteilung(
  qabt_id               SERIAL          PRIMARY KEY,
  qabt_bez              VARCHAR(80)     NOT NULL,
  qabt_isService        BOOLEAN         DEFAULT FALSE,
  qabt_inaktiv          BOOLEAN NOT NULL DEFAULT FALSE
);

CREATE OR REPLACE FUNCTION qababteilung__a_iud() RETURNS TRIGGER AS $$               -- #7018
  BEGIN
    IF NOT tg_op='INSERT' THEN
        DELETE FROM qababteilungtxt WHERE qabtl_qabt_id=old.qabt_id AND qabtl_spr_key=prodat_languages.current_lang();
    END IF;
    IF NOT tg_op='DELETE' THEN
        INSERT INTO qababteilungtxt (qabtl_qabt_id, qabtl_spr_key, qabtl_txt) VALUES (new.qabt_id, prodat_languages.current_lang(), new.qabt_bez);
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER qababteilung__a_iud                                                 -- #7018
  AFTER INSERT OR UPDATE OR DELETE
  ON qababteilung
  FOR EACH ROW
  EXECUTE PROCEDURE qababteilung__a_iud();

CREATE TABLE qababteilungtxt                                                         -- #7018
 (qabtl_id               SERIAL NOT NULL PRIMARY KEY,
  qabtl_qabt_id          INTEGER NOT NULL REFERENCES qababteilung ON DELETE CASCADE,
  qabtl_spr_key          VARCHAR(5) NOT NULL REFERENCES sprach,
  qabtl_txt              VARCHAR(80)
 );

CREATE OR REPLACE FUNCTION lang_qababteilung(INTEGER, VARCHAR) RETURNS VARCHAR AS $$     -- #7018
DECLARE result VARCHAR;
  BEGIN
   SELECT COALESCE(qabtl_txt,qabt_bez) INTO result
    FROM qababteilung
      LEFT JOIN qababteilungtxt ON qabtl_qabt_id=qabt_id AND qabtl_spr_key=$2
   WHERE qabt_id=$1;
  RETURN result;
END $$ LANGUAGE plpgsql;


--Vorgaben für Sofortmaßnahmen eines QAB
/*Sofortmaßnahmen in Fremdsprachen */
CREATE TABLE qabsofort(
  qsof_id               SERIAL          PRIMARY KEY,
  qsof_bez              VARCHAR(80)     NOT NULL,
  qsof_dim              VARCHAR(2),
  qsof_isService        BOOLEAN         DEFAULT FALSE,
  qsof_inaktiv          BOOLEAN NOT NULL DEFAULT FALSE
);

CREATE OR REPLACE FUNCTION qabsofort__a_iud() RETURNS TRIGGER AS $$                  -- #7018
  BEGIN
    IF NOT tg_op='INSERT' THEN
        DELETE FROM qabsoforttxt WHERE qsofl_qsof_id=old.qsof_id AND qsofl_spr_key=prodat_languages.current_lang();
    END IF;
    IF NOT tg_op='DELETE' THEN
        INSERT INTO qabsoforttxt (qsofl_qsof_id, qsofl_spr_key, qsofl_txt) VALUES (new.qsof_id, prodat_languages.current_lang(), new.qsof_bez);
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER qabsofort__a_iud                                                    -- #7018
  AFTER INSERT OR UPDATE OR DELETE
  ON qabsofort
  FOR EACH ROW
  EXECUTE PROCEDURE qabsofort__a_iud();

CREATE TABLE qabsoforttxt                                                            -- #7018
 (qsofl_id               SERIAL NOT NULL PRIMARY KEY,
  qsofl_qsof_id          INTEGER NOT NULL REFERENCES qabsofort ON DELETE CASCADE,
  qsofl_spr_key          VARCHAR(5) NOT NULL REFERENCES sprach,
  qsofl_txt              VARCHAR(80)
 );


CREATE OR REPLACE FUNCTION lang_qabsofort(INTEGER, VARCHAR) RETURNS VARCHAR AS $$    -- #7018
DECLARE result VARCHAR;
  BEGIN
   SELECT COALESCE(qsofl_txt,qsof_bez) INTO result
    FROM qabsofort
      LEFT JOIN qabsoforttxt ON qsofl_qsof_id=qsof_id AND qsofl_spr_key=$2
   WHERE qsof_id = $1;
  RETURN result;
END $$ LANGUAGE plpgsql;

-- #19744 QAB: Oberflächenkonfiguration
CREATE TABLE qab_gui_config (
  qgc_id                        serial NOT null PRIMARY KEY,      -- technischer Primärschlüssel
  qgc_bez_xtt                   integer NOT null,                 -- Konfigurationsbezeichnung als text0-Index, Text wird in der GUI angezeigt
  qgc_pos                       integer NOT null,                 -- Rangordnung der Konfig, wie in der GUI im LookUpEdit angezeigt
  qgc_standard                  boolean NOT null DEFAULT false,   -- Handelt es sich um die Standardkonfiguration?
  qgc_d2_visible                boolean NOT null DEFAULT true,    -- Tab "D2" sichtbar?
  qgc_d3_visible                boolean NOT null DEFAULT true,    -- Tab "D3" sichtbar?
  qgc_d4_visible                boolean NOT null DEFAULT true,    -- Tab "D4" sichtbar?
  qgc_d5_visible                boolean NOT null DEFAULT true,    -- Tab "D5" sichtbar?
  qgc_d6_visible                boolean NOT null DEFAULT true,    -- Tab "D6" sichtbar?
  qgc_d7_visible                boolean NOT null DEFAULT true,    -- Tab "D7" sichtbar?
  qgc_d8_visible                boolean NOT null DEFAULT true,    -- Tab "D8" sichtbar?
  qgc_bem                       text                              -- Bemerkung als Freitext
);

-- Konfigbezeichnung muss eindeutig sein.
CREATE UNIQUE INDEX qab_gui_config__qgc_bez_xtt__uq ON qab_gui_config( qgc_bez_xtt );

-- Konfigposition muss eindeutig sein.
CREATE UNIQUE INDEX qab_gui_config__qgc_pos__uq ON qab_gui_config( qgc_pos );

-- Es darf maximal eine Standardkonfiguration geben.
CREATE UNIQUE INDEX qab_gui_config__qgc_standard__uq ON qab_gui_config( qgc_standard ) WHERE qgc_standard;


---------------------    Vorgaben QA-Berichte Ende  -----------------------------------

--Qualitätsabweichungsberichte
CREATE SEQUENCE qab_q_nr_seq;
CREATE TABLE qab(
  q_nr                  INTEGER PRIMARY KEY,                            --Nummer des Qualitätsberichts
  -- QAB-Typ. 1=Frei,2=Artikelbezogen,3=Wareneingang,4=Lagerabgang,5=Fertigungsauftrag,6=Servicevorfall, 7=Vertrag / Wartungsvertrag
  q_typ                 INTEGER NOT NULL,
  q_q_nr                INTEGER REFERENCES qab    ON UPDATE CASCADE,    -- QAB - Bezug. z.Bsp. für Wiederholungsfehler
  q_w_wen               INTEGER REFERENCES wendat ON UPDATE CASCADE,    -- Wareneingang der bemängelt wurde
  q_l_nr                INTEGER REFERENCES lifsch ON UPDATE CASCADE,    -- Warenausgang der bemängelt wurde
  q_ab_ix               INTEGER REFERENCES abk ON UPDATE CASCADE,       -- ABK von fehlerhafter Produktion
  q_a2_n                INTEGER,                                        -- Arbeitsgang der ABK
  q_awr_lifschnr        VARCHAR(20),                                    -- Auswärtsbearbeitung (für Lieferschein Rücklieferung)
  q_ak_nr               VARCHAR(40) REFERENCES art ON UPDATE CASCADE,   -- Fehlerhafter Artikel
  q_kanfse_id           INTEGER REFERENCES kundservicepos ON UPDATE CASCADE,  --QAB ist an diese Serviceanfrage gebunden
  q_vtr_nr              VARCHAR(40) REFERENCES vertrag,                 --Gebunden an Vertrag
  -- Status-Flags
  q_def                 BOOLEAN NOT NULL DEFAULT FALSE,                 -- Definitiv-Status
  q_reklamok            BOOLEAN DEFAULT FALSE,                          -- Die Reklamation/der QAB war berechtigt, Fehler auf Seiten der Firma
  q_reklamnotok         BOOLEAN DEFAULT FALSE,                          -- Die Reklamation/der QAB war unberechtigt (ausdrücklich separates Feld gewünscht)
  q_done                BOOLEAN DEFAULT False,                          -- Bearbeitung des QAB ist abgeschlossen. QAB-Leiter kann den definitiv setzen
  q_wirksam             BOOLEAN DEFAULT FALSE,                          -- QAB-Maßnahme ist wirksam, sprich: Es hat was gebracht / wird demnächst anders gemacht. Das ist NICHT das Gleiche wie definitiv.
  q_isService           BOOLEAN NOT NULL DEFAULT FALSE,                 -- Zur Unterscheidung von Sericevorgang und QS-Vorgang.
  q_isPReport           BOOLEAN NOT NULL DEFAULT FALSE,                 -- Zur Unterscheidung von Problemreport und QS-Vorgang.
  q_rech_sperre         BOOLEAN NOT NULL DEFAULT FALSE,                 -- Markiert zugehörige Rechnung(en), derzeit nur Eingangsseitig, damit Zahlung ggf. verzögert werden kann
  --
  q_dat                 DATE,                                           -- Datum der Erfassung des QAB
  q_referenz            VARCHAR(50),                                    -- freie Referenznummer, z.B. Prüfberichtsnummer des Kunden
  q_krzl_apkrzl         VARCHAR(10),                                    -- Ansprechpartner Adressant Kurzname
  q_krzl_ap             VARCHAR(75),                                    -- Ansprechpartner Adressant
  q_krzl                VARCHAR(21) REFERENCES adk ON UPDATE CASCADE,   -- Kurname Adressant
  q_api                 VARCHAR(20),                                    -- Ansprechpartner Intern (ll_db_usename)
  q_adresstxt           TEXT,                                           -- Zusatztext für Adressfeld für Hinweise, sonstige Kontaktinfo etc
  q_festext             BOOLEAN NOT NULL DEFAULT FALSE,                 -- True = Fehler wurde extern festgestellt
  q_kost_br             NUMERIC(12,4) NOT NULL DEFAULT 0,               -- Bruttokosten
  q_kost_rueck          NUMERIC(12,4) NOT NULL DEFAULT 0,               -- Rückerstattete Kosten
  q_kost_nt             NUMERIC(12,4) NOT NULL DEFAULT 0,               -- Verursachte Kosten netto
  q_txt                 TEXT,                                           -- Freie Bemerkungen
  q_txt_rtf             TEXT,
  q_fehtxt              TEXT,                                           --Fehlerbeschreibung
  q_fehtxt_rtf          TEXT,
  q_fehdat              DATE,
  q_fehvon              VARCHAR(10),
  q_stelltxt            TEXT,                                           --Stellungnahme
  q_stelltxt_rtf        TEXT,
  q_stelldat            DATE,
  q_stellvon            VARCHAR(10),
  q_bestxt              TEXT,                                           --Fehlerbeseititung
  q_bestxt_rtf          TEXT,
  q_besdat              DATE,
  q_besvon              VARCHAR(10),
  q_prueftxt            TEXT,                                           --Überprüfung der Nacharbeit
  q_prueftxt_rtf        TEXT,
  q_pruefdat            DATE,
  q_pruefvon            VARCHAR(10),
  q_bewtxt              TEXT,                                           --Kosten/Auswertung
  q_bewtxt_rtf          TEXT,
  q_bewdat              DATE,
  q_bewvon              VARCHAR(10),
  q_ruecktxt            TEXT,                                           --Auswertung/Rückerstattung
  q_ruecktxt_rtf        TEXT,
  q_rueckdat            DATE,
  q_rueckvon            VARCHAR(10),
  q_wirktxt             TEXT,                                           --Wirksamkeitsprüfung der Maßnahme
  q_wirktxt_rtf         TEXT,
  q_wirkdat             DATE,
  q_wirkvon             VARCHAR(10),
  q_vorbtxt             TEXT,                                           --Vorbeugungsmaßnahmen
  q_vorbtxt_rtf         TEXT,
  q_vorbdat             DATE,
  q_vorbvon             VARCHAR(10),
  q_korrtxt             TEXT,                                           --Korrekturmaßnahmen
  q_korrtxt_rtf         TEXT,
  q_korrdat             DATE,
  q_korrvon             VARCHAR(10),
  q_smass_txt           TEXT,                                           --Sofortmaßnahmen
  q_smass_txt_rtf       TEXT,
  q_smassdat            DATE,
  q_smassvon            VARCHAR(10),
  q_feh_id              INTEGER REFERENCES qabfehler ON UPDATE CASCADE, --Fehlerart
  q_ent_id              INTEGER REFERENCES qabentscheid ON UPDATE CASCADE, --Getroffene Maßnahme
  q_kla_id              INTEGER REFERENCES qabklasse ON UPDATE CASCADE,    --Einordnung des QAB
  q_abt_id              INTEGER REFERENCES qababteilung ON UPDATE CASCADE, --Verantwortliche Abteilung
  q_verursacher         VARCHAR(50),                                       --direkter verursacher (freifeld) zB Kurzname Mitarbeiter
  --Qualitätsabweichungsbericht: freie Fleder
  q_frei1               VARCHAR(21)     REFERENCES adk ON UPDATE CASCADE ON DELETE SET NULL,-- Adresskrz.
  q_frei2               VARCHAR(50),       --Firma
  q_frei3               VARCHAR(50),       --Artikel
  q_frei4               VARCHAR(50),       --Mengeneinheit
  q_frei5               VARCHAR(50),       --Lieferschein
  q_frei6               VARCHAR(50),       --Auftragsnummer
  q_frei7               VARCHAR(50),       --Kunde
  q_freimenge           NUMERIC(12,2),     --Menge
  --Servicevorfall: freie Felder
  q_frei8               VARCHAR(50),       --alphanumerisch 1 (qab)
  q_frei9               VARCHAR(50),       --alphanumerisch 2 (qab)
  q_frei10              VARCHAR(50),       --alphanumerisch 3 (qab)
  q_frei11              VARCHAR(50),       --alphanumerisch 4 (qab)
  q_freimenge2          NUMERIC(12,2),     --numerisch 1 (qab)
  q_freimenge3          NUMERIC(12,2),     --numerisch 2 (qab)
  -- unused_q_istfrei           BOOLEAN DEFAULT FALSE,      -- QAB wurde frei erstellt, ab V12.1 über q_typ abgebildet
  q_sernr               TEXT,                             --Seriennummern (als Vorgabe im Lagerzugang Kundenreklamation)
  q_maengelanz          TEXT,                             --Text für Brief "Mängelanzeige"
  q_maengelanz_rtf      TEXT,
  q_stk_fehlerhaft      NUMERIC(12,2) NOT NULL DEFAULT 0, --Anzahl fehlerhafter Teile
  q_stk                 NUMERIC(12,2) NOT NULL DEFAULT 0, --Gesamtzahl der Teile, auf die sich QAB bezieht.
  q_mecode              INTEGER NOT NULL DEFAULT 1 REFERENCES mgcode, --In welcher Mengeneinheit.
  q_qgc_id              integer REFERENCES qab_gui_config ON DELETE SET null  -- #19744 Verbindung zur QAB-Oberflächenkonfiguration
);

CREATE INDEX qab_q_w_wen ON qab (q_w_wen);
CREATE INDEX qab_q_ak_nr ON qab (q_ak_nr);
CREATE INDEX qab_q_ak_nr_like ON qab (q_ak_nr  varchar_pattern_ops);
--
CREATE TRIGGER qab_delete_abk_project_structure
 AFTER DELETE ON qab
 FOR EACH ROW
 EXECUTE PROCEDURE table_delete_abkstru();
--

--InfoNachrichten für neue QABs
CREATE OR REPLACE FUNCTION qab__b_iu() RETURNS TRIGGER AS $$
  BEGIN
    --Nur SYS.QAB-Admins dürfen QAB wirksam setzen
    IF (tg_op='UPDATE') THEN
        IF  (NOT new.q_isService) AND (new.q_def <> old.q_def) THEN
                IF NOT current_user IN (SELECT usename FROM pg_group, pg_user WHERE usesysid=ANY(grolist) AND groname='SYS.QAB-Admin') THEN
                        RAISE EXCEPTION 'xtt10261';
                END IF;
        END IF;
    END IF;

    --Wirksam setzen = Definitiv setzen
    IF (new.q_wirksam) THEN
        new.q_def:=TRUE;
    END IF;

    -- Wareneingangsgebunden und Artikelnummer vorhanden ...
    IF (new.q_w_wen IS NOT NULL) AND ( NullIF(new.q_ak_nr,'') IS NULL ) THEN
      -- Keine Rücklieferung auf Auswärtsbearbeitung, WE-Artikel nehmen. Bei Auswärtsbearbeitung würde da das Arbeitspaket stehen das wir nicht nehmen können.
      IF NullIF(new.q_awr_lifschnr,'') IS NULL THEN
        new.q_ak_nr:= w_aknr FROM wendat WHERE w_wen = new.q_w_wen;
      END IF;
    END IF;

    IF (new.q_l_nr IS NOT NULL) AND (NullIF(new.q_ak_nr,'') IS NULL) THEN
        new.q_ak_nr:= l_aknr FROM lifsch WHERE l_nr = new.q_l_nr;
    END IF;

    -- Nur wenn noch keine Ak-Nr angegeben
    IF (new.q_ab_ix IS NOT NULL) AND (NullIF(new.q_ak_nr,'') IS NULL) THEN
        new.q_ak_nr:=COALESCE((tableident_allfields(ab_tablename, ab_dbrid)).aknr, ak_nr) AS aknr
            FROM  abk LEFT JOIN ldsdok ON ld_abk=ab_ix LEFT JOIN art ON COALESCE(ab_ap_nr, ld_aknr)=ak_nr
            WHERE ab_ix=new.q_ab_ix
            ORDER BY ab_pos, ab_parentabk IS NOT NULL, NOT ab_ix<ab_parentabk, ab_ix
            LIMIT 1;
        --op_n FROM abk LEFT OUTER JOIN opl ON ab_askix = op_ix WHERE ab_ix = new.q_ab_ix;
    END IF;



    --Wenn ABK angegeben, prüfen ob Arbeitsgang existiert
    IF (new.q_ab_ix IS NOT NULL) AND (new.q_a2_n IS NOT NULL)
        AND ((SELECT a2_id FROM ab2 WHERE a2_ab_ix=new.q_ab_ix AND a2_n = new.q_a2_n) IS NULL) THEN
        RAISE EXCEPTION 'xtt10259';
    END IF;

    new.q_kost_nt:=COALESCE(new.q_kost_br,0)-COALESCE(new.q_kost_rueck,0);

    IF (tg_op='INSERT') THEN

        IF new.q_mecode = 0 THEN -- Wenn ein QAB ohne Artikel angelegt wird, trifft die Suche nach Standard-ME des Artikels nicht.
          new.q_mecode = 1;      -- Sollte das nicht in der Oberfläche abgefangen sein, kann ein me_code=0 bis hierhin durchrutschen,
                                 -- was dann Fehler verursacht. Sicherheitshalber wird das also hier abgefangen.
        END IF;

        -- QAB auf Wareneingang => Nachricht an QAB-Extern-Neu-Infomail
        IF (new.q_w_wen IS NOT NULL) THEN

                PERFORM
                 msg_SendMsg(CAST(current_user AS VARCHAR), 'Neuer QAB '||new.q_nr || ' (Wareneingang) ', '', CAST(usename AS VARCHAR))
                FROM
                 pg_group,
                 pg_user
                WHERE
                 (groname='SYS.QAB-Extern-Neu-Infomail')
                AND
                usesysid=ANY(grolist);

                --Nachricht an SYS.QAB-Neu-Infomail, aber nur, falls nicht schon bei erster Gruppe dabei
                PERFORM
                 msg_SendMsg(CAST(current_user AS VARCHAR), 'Neuer QAB '||new.q_nr || ' (Wareneingang) ', '', CAST(usename AS VARCHAR))
                FROM
                 pg_group,
                 pg_user
                WHERE
                 (groname='SYS.QAB-Neu-Infomail')
                AND usename NOT IN      (SELECT USENAME
                                         FROM PG_GROUP,PG_USER
                                         WHERE (GRONAME='SYS.QAB-Extern-Neu-Infomail') AND usesysid=ANY(grolist)
                                        )
                AND
                 usesysid=ANY(grolist);


        -- QAB auf Auswärtsbearbeitung => Nachricht an QAB-Auswaerts-Neu-Infomail
        ELSEIF (new.q_awr_lifschnr IS NOT NULL) THEN

                PERFORM
                 msg_SendMsg(CAST(current_user AS VARCHAR), 'Neuer QAB '||new.q_nr|| ' (Auswaertsbearbeitung)', '', CAST(usename AS VARCHAR))
                FROM
                 pg_group,
                 pg_user
                WHERE
                 (groname='SYS.QAB-Auswaerts-Neu-Infomail')
                AND
                 usesysid=ANY(grolist);

                --Nachricht an SYS.QAB-Neu-Infomail, aber nur, falls nicht schon bei erster Gruppe dabei
                PERFORM
                 msg_SendMsg(CAST(current_user AS VARCHAR), 'Neuer QAB '||new.q_nr || ' (Auswaertsbearbeitung) ', '', CAST(usename AS VARCHAR))
                FROM
                 pg_group,
                 pg_user
                WHERE
                 (groname='SYS.QAB-Neu-Infomail')
                AND usename NOT IN      (SELECT USENAME
                                         FROM PG_GROUP,PG_USER
                                         WHERE (GRONAME='SYS.QAB-Auswaerts-Neu-Infomail') AND usesysid=ANY(grolist)
                                        )
                AND
                 usesysid=ANY(grolist);

        -- Sonstiger QAB (Kundenreklamation oder frei) => Nachricht nur an 'SYS.QAB-Neu-Infomail'
        ELSE
                PERFORM
                 msg_SendMsg(CAST(current_user AS VARCHAR), 'Neuer QAB '||new.q_nr, '', CAST(usename AS VARCHAR))
                FROM
                 pg_group,
                 pg_user
                WHERE
                 (groname='SYS.QAB-Neu-Infomail')
                AND
                 usesysid=ANY(grolist);
        END IF;
    END IF;


    --bei Änderung Wareneingangsnummer Nachricht an alle die eingetragen sind in Gruppe
    IF tg_op='UPDATE' THEN
        IF COALESCE(new.q_w_wen,0)<>COALESCE(old.q_w_wen,0) THEN
        PERFORM
         msg_SendMsg(CAST(current_user AS VARCHAR),
                'QAB '||new.q_nr||' bearbeitet',
                'Neuer Wareneingang zugewiesen (WE-Nr: ' || COALESCE(CAST(new.q_w_wen AS VARCHAR),'Keine') || ')',
                CAST(usename AS VARCHAR))
        FROM
         pg_group,
         pg_user
        WHERE
         groname='SYS.QAB-Extern-Neu-Infomail'
        AND
         usesysid=ANY(grolist);
        END IF;
    END IF;


    --bei Änderung Rücklieferung für Auswaertsbearbeitung Nachricht an alle die eingetragen sind in Gruppe
    IF tg_op='UPDATE' THEN
        IF COALESCE(new.q_awr_lifschnr,'')<>COALESCE(old.q_awr_lifschnr,'') THEN
        PERFORM
         msg_SendMsg(CAST(current_user AS VARCHAR),
                'QAB '||new.q_nr||' bearbeitet',
                'Neue Ruecklieferung aus Auswaertsbearbeitung zugewiesen (Liefer-Nr.Auswaerts: ' || COALESCE(CAST(new.q_awr_lifschnr AS VARCHAR),'Keine') || ')',
                CAST(usename AS VARCHAR))
        FROM
         pg_group,
         pg_user
        WHERE
         groname='QAB-Auswaerts-Neu-Infomail'
        AND
         usesysid=ANY(grolist);
        END IF;
    END IF;

    --bei Änderung Lagerabgangs-Nr. Nachricht an alle die eingetragen sind in Gruppe
    IF tg_op='UPDATE' THEN
        IF COALESCE(new.q_l_nr,0)<>COALESCE(old.q_l_nr,0) THEN
                PERFORM
                 msg_SendMsg(CAST(current_user AS VARCHAR),
                        'QAB - Ruecklieferschein '||(SELECT beld_dokunr
                                                 FROM lieferschein
                                                   JOIN lieferschein_pos ON belp_dokument_id = beld_id
                                                   JOIN lifsch ON belp_id = l_belp_id
                                                 WHERE l_nr=new.q_l_nr ORDER BY l_nr DESC LIMIT 1),
                        'Neuer Warenausgang zugewiesen:'||E'\n'||'Warenausgangs-Nr: '||new.q_l_nr||E'\n'||'QAB-Nr: '||new.q_nr,
                        CAST(usename AS VARCHAR))
                FROM
                 pg_group,
                 pg_user
                WHERE
                 groname='SYS.QAB-Retoure-Infomail'
                AND
                 usesysid=ANY(grolist);
        END IF;
    END IF;

    --bei Wirksam setzen Nachricht an alle die eingetragen sind in Gruppe
    IF tg_op='UPDATE' THEN
        If (new.q_wirksam) AND (NOT old.q_wirksam) THEN
                PERFORM
                 msg_SendMsg(CAST(current_user AS VARCHAR),
                        'QAB - Wirksam gesetzt: '||(new.q_nr),
                        '',
                        CAST(usename AS VARCHAR))
                FROM
                 pg_group,
                 pg_user
                WHERE
                 groname='SYS.QAB-Wirksam-Infomail'
                AND
                 usesysid=ANY(grolist);
        END IF;
    END IF;

    RETURN new;
END$$LANGUAGE plpgsql;
--
CREATE TRIGGER qab__b_iu
  BEFORE INSERT OR UPDATE
  ON qab
  FOR EACH ROW
EXECUTE PROCEDURE qab__b_iu();

----------------------------

--Statuskennzeichen 'QAB' = 'QAB-Auslöser' #10072
CREATE OR REPLACE FUNCTION qab__a_iu_abk_status() RETURNS TRIGGER AS $$
BEGIN
  --entfernen, wenn die verknüpfte ABK sich geändert hat
  IF (tg_op = 'UPDATE') AND (old.q_ab_ix IS NOT NULL) THEN
    UPDATE abk SET ab_stat = TSystem.ENUM_delvalue(ab_stat, 'QAB') WHERE old.q_ab_ix = ab_ix; -- ??? Wurde vorher null gesetzt der AB_STAT
  END IF;
  --Status setzen wenn q_ab_ix gesetzt wurde
  UPDATE abk SET ab_stat = TSystem.ENUM_SetValue(ab_stat,'QAB') WHERE new.q_ab_ix = ab_ix;  -- 'QAB-Auslöser'
  --
  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER qab__a_iu_abk_status
  AFTER INSERT OR UPDATE OF q_ab_ix
  ON qab
  FOR EACH ROW
EXECUTE PROCEDURE qab__a_iu_abk_status();


-- Legt beim Insert die Default-Parameter für QAB/ Problembericht an
CREATE TRIGGER qab__a_i__create_autoparams
  AFTER INSERT
  ON qab
  FOR EACH ROW
EXECUTE PROCEDURE TRecnoParam.CreateAutoParams();

-- Funktion zur Status-Einordnung eines Problemberichts auf Basis der gesetzten Parameter
CREATE OR REPLACE FUNCTION GetPrReportStatus(IN prnr integer, OUT neu boolean, OUT _po boolean, OUT _do boolean, OUT einstwert boolean, OUT einstbez character varying, OUT mnpo boolean, OUT mndo boolean, OUT prlaufzeit integer, OUT qmgemeldet boolean, OUT entwurf boolean, OUT ergpo boolean, OUT ergdo boolean, OUT ursanalyse boolean) AS $$
DECLARE rdbrid VARCHAR;
BEGIN
  SELECT dbrid FROM qab WHERE q_nr=PrNr INTO rdbrid;
  IF NOT FOUND THEN
    --RAISE EXCEPTION '%', Format(lang_text(29179) /*'PR-Nummer % ist nicht gefunden.'*/, PrNr);
    RETURN;
  END IF;
  _po:=TRecNoParam.GetBool('qab.d4.p01', rdbrid);
  _do:=TRecNoParam.GetBool('qab.d4.p04', rdbrid);
  QMGemeldet:=TRecNoParam.GetBool('qab.custom.qmgemeldet', rdbrid);
  --wenn von PO oder DO noch nicht entschieden ist und bei 2.1 nichts angekreuzt
  IF QMGemeldet AND ((_do IS NULL) OR (_po IS NULL)) AND (NOT EXISTS(SELECT TRUE FROM recnokeyword WHERE r_dbrid = rdbrid AND r_reg_pname ~* E'^qab\\.d3\\.2\\.p\\d+$' AND lower(r_value)='true')) THEN
    neu:=TRUE;
  ELSE
    neu:=FALSE;
  END IF;
  /* === Einstufung ===
  'Kein Einfluss auf Sicherheit'    qab.d4.p05 - qab.d4.p10 'keine Meldung'
  'geringfügig'                     qab.d4.p06 - qab.d4.p11 'Meldung an TCH'
  'bedeutend'                       qab.d4.p07 - qab.d4.p12 'Meldung an TCH*, ECCAIRS'
  'gefährlich'                      qab.d4.p08 - qab.d4.p13 'Meldung an IORS (EASA), LBA (Form44), TCH*, ECCAIRS gem Anhang A der VA-AL-003'
  'katastrophal'                    qab.d4.p09 - qab.d4.p14 'Meldung an IORS (EASA), LBA (Form44), TCH*, ECCAIRS gem Anhang A der VA-AL-003,'
  */
  IF TRecNoParam.GetBool('qab.d4.p05', rdbrid) THEN
    einstbez:=TRecNoParam.GetBez('qab.d4.p05') || ' - ' || TRecNoParam.GetBez('qab.d4.p10');
    einstwert:=TRecNoParam.GetBool('qab.d4.p10', rdbrid);
  ELSEIF TRecNoParam.GetBool('qab.d4.p06', rdbrid) THEN
    einstbez:=TRecNoParam.GetBez('qab.d4.p06') || ' - ' || TRecNoParam.GetBez('qab.d4.p11');
    einstwert:=TRecNoParam.GetBool('qab.d4.p11', rdbrid);
  ELSEIF TRecNoParam.GetBool('qab.d4.p07', rdbrid) THEN
    einstbez:=TRecNoParam.GetBez('qab.d4.p07') || ' - ' || TRecNoParam.GetBez('qab.d4.p12');
    einstwert:=TRecNoParam.GetBool('qab.d4.p12', rdbrid);
  ELSEIF TRecNoParam.GetBool('qab.d4.p08', rdbrid) THEN
    einstbez:=TRecNoParam.GetBez('qab.d4.p08') || ' - ' || TRecNoParam.GetBez('qab.d4.p13');
    einstwert:=TRecNoParam.GetBool('qab.d4.p13', rdbrid);
  ELSEIF TRecNoParam.GetBool('qab.d4.p09', rdbrid) THEN
    einstbez:=TRecNoParam.GetBez('qab.d4.p09') || ' - ' || TRecNoParam.GetBez('qab.d4.p14');
    einstwert:=TRecNoParam.GetBool('qab.d4.p14', rdbrid);
  ELSE
    einstbez:='OFFEN';
    einstwert:=NULL;
  END IF;

  --Maßnahmen PO: Wenn ein Parameter von qab.d5.1.p01 - qab.d5.1.p06 eingecheckt ist -> TRUE, sonnst FALSE
  mnPO:=EXISTS(SELECT TRUE FROM recnokeyword WHERE r_dbrid = rdbrid AND r_reg_pname ~* E'^qab\\.d5\\.1\\.p0[1-6]{1}$' AND lower(r_value)='true');

  --Maßnahmen DO: Wenn ein Parameter von qab.d5.2.p07 - qab.d5.2.p10 eingecheckt ist -> TRUE, sonnst FALSE
  mnDO:=EXISTS(SELECT TRUE FROM recnokeyword WHERE r_dbrid = rdbrid AND r_reg_pname ~* E'^qab\\.d5\\.2\\.p(?:(0[7-9]{1})|(10))$' AND lower(r_value)='true');

  --Ergebnisse DO: Wenn ein Parameter von qab.d6.p01 - qab.d6.p06 eingecheckt ist -> TRUE, sonnst FALSE
  ergDO:=EXISTS(SELECT TRUE FROM recnokeyword WHERE r_dbrid = rdbrid AND r_reg_pname ~* E'^qab\\.d6\\.p0[1-6]{1}$' AND r_value IS NOT NULL);

  --Ergebnisse PO: Wenn ein Parameter von qab.d7.p01 - qab.d7.p07 eingecheckt ist -> TRUE, sonnst FALSE
  ergPO:=EXISTS(SELECT TRUE FROM recnokeyword WHERE r_dbrid = rdbrid AND r_reg_pname ~* E'^qab\\.d7\\.p0[1-7]{1}$' AND r_value IS NOT NULL);

  --Ursachenanalyse: Wenn ein Parameter von qab.d8.pXX eingecheckt ist -> TRUE, sonnst FALSE
  ursAnalyse:=EXISTS(SELECT TRUE FROM recnokeyword WHERE r_dbrid = rdbrid AND r_reg_pname ~* E'^qab\\.d8\\.p\\d+$' AND lower(r_value)='true');

  --Differenz in Tagen zwischen Erstelldatum und HEUTE
  PRLaufzeit:=current_date - (SELECT q_dat FROM qab WHERE dbrid=rdbrid);

  RETURN;
END $$ LANGUAGE plpgsql STABLE;
----------------------------

CREATE OR REPLACE FUNCTION qab__a_iu_UpdateWendat() RETURNS TRIGGER AS $$
BEGIN
  --Wenn QAB auf Wareneingang erstellt wird, markieren wir im Wareneingang, welcher QAB das ist.
  UPDATE wendat SET w_q_nr = new.q_nr WHERE w_wen = new.q_w_wen;
  RETURN new;
END$$LANGUAGE plpgsql;

CREATE TRIGGER qab__a_iu_UpdateWendat
  AFTER INSERT OR UPDATE
  OF q_w_wen
  ON qab
  FOR EACH ROW
EXECUTE PROCEDURE qab__a_iu_UpdateWendat();

-- #8468 Nachsetzen der Verknüpfung von Angebot/Auftrag mit Servicevorfall: Wenn erst Angebot und dann Servicevorfall angelegt wird
CREATE OR REPLACE FUNCTION qab__a_i_auftg_serviceLink() RETURNS TRIGGER AS $$
BEGIN
  UPDATE auftg SET ag_q_nr = new.q_nr WHERE ag_id = (SELECT kanfse_ag_id FROM kundservicepos WHERE kanfse_id = new.q_kanfse_id);
  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER qab__a_i_auftg_serviceLink
  AFTER INSERT
  ON qab
  FOR EACH ROW
  WHEN (new.q_kanfse_id IS NOT NULL)
EXECUTE PROCEDURE qab__a_i_auftg_serviceLink();
---

--- #17906 [Service] Automatisches Laden von Arbeitspaket, Vorlagestruktur beim Erstellen eines SVs
CREATE OR REPLACE FUNCTION qab__a_90_i__service_abk() RETURNS TRIGGER AS $$
BEGIN
  PERFORM tabk.abk__create(op_n, op_ix, 1, _ab_tablename => 'qab', _ab_dbrid => new.dbrid) FROM opl WHERE op_ix = TSystem.Settings__Get('ASK__ALS__SERVICE_ABK__ANLEGEN');
  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER qab__a_90_i__service_abk
  AFTER INSERT
    ON qab
  FOR EACH ROW
      WHEN ( (new.q_isService) AND (TSystem.Settings__Get('ASK__ALS__SERVICE_ABK__ANLEGEN') IS NOT null) )
      EXECUTE PROCEDURE qab__a_90_i__service_abk();
---

---- #20041 [Service] Automatisches Laden von Arbeitspaket, Vorlagestruktur beim Erstellen eines SVs
---- Alle QABs ohne GUI-Vorgaben bekommen die Standard-GUI-Vorgaben, auch die Service-QABs.
CREATE OR REPLACE FUNCTION qab__b_90_i__gui_config() RETURNS TRIGGER AS $$
DECLARE
  _standard_id integer;
BEGIN
  _standard_id := qgc_id 
    FROM qab_gui_config 
    ORDER BY qgc_standard DESC, qgc_id ASC 
    LIMIT 1;

  -- Service-QABs haben immer die Standard-GUI.
  IF _standard_id IS NOT null OR ( new.q_typ = 6 AND new.q_qgc_id IS DISTINCT FROM _standard_id ) THEN
    new.q_qgc_id = _standard_id;
  END IF;

  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER qab__b_90_i__gui_config
  BEFORE INSERT OR UPDATE OF q_qgc_id, q_typ
  ON qab
  FOR EACH ROW
  WHEN ( new.q_qgc_id IS null OR new.q_typ = 6 ) -- 6 bedeutet Service-QAB
  EXECUTE PROCEDURE qab__b_90_i__gui_config();
---

--Maßnahmen für einen bestimmten QAB
CREATE TABLE qabmassnahmen(
   qmas_id      SERIAL           PRIMARY KEY,
   qmas_qnr     INTEGER NOT NULL REFERENCES qab ON UPDATE CASCADE ON DELETE CASCADE,
   qmas_sofid   INTEGER NOT NULL REFERENCES qabsofort ON UPDATE CASCADE ON DELETE CASCADE,
   --#5630 QS-Modul, Erweiterung Überwachung
   qmas_bearbeiter INTEGER REFERENCES llv ON DELETE SET NULL,    -- Bearbeiter
   qmas_termin DATE DEFAULT now(),                               -- Termin
   qmas_done_date DATE                                           -- Beendet am
);


--QAB-Team zur Betreuung eines QAB
CREATE TABLE qabteam (
   qteam_id         SERIAL PRIMARY KEY,
   qteam_qnr        INTEGER NOT NULL REFERENCES qab ON UPDATE CASCADE ON DELETE CASCADE,
   qteam_minr       VARCHAR(10),      -- Ist ll_db_usename, 2013 umgeschrieben
   qteam_leiter     BOOLEAN NOT NULL DEFAULT FALSE,
   qteam_funktion   VARCHAR(10)    REFERENCES anl_pfunktion
);

--Jeder Mitarbeiter kann nur einmal im QAB Team sein
CREATE UNIQUE INDEX qabteam_qnr_minr ON qabteam(qteam_qnr,qteam_minr);

-- Sicherstellen, das nur ein Mitglied Teamleiter ist
CREATE OR REPLACE FUNCTION qabteam__b_iu() RETURNS TRIGGER AS $$
  BEGIN

    IF new.qteam_leiter THEN
      UPDATE qabTeam SET qteam_leiter = FALSE WHERE qteam_qnr = new.qteam_qnr AND qteam_leiter AND qteam_id <> new.qteam_id;
    END IF;

    RETURN new;
 END$$LANGUAGE plpgsql;
--
CREATE TRIGGER qabteam__b_iu
  BEFORE INSERT OR UPDATE
  ON qabteam
  FOR EACH ROW
 EXECUTE PROCEDURE qabteam__b_iu();

--Wareneingangskontrolle - Vorgaben für Fehlerklassen
CREATE TABLE wekfehlerklasse
 (
   wef_id       SERIAL          PRIMARY KEY,
   wef_bez      VARCHAR(80)     NOT NULL
);

--Wareneingangskontrolle - Vorgaben für zu ergreifende Maßnahmen
CREATE TABLE weksofort(
   wes_id       SERIAL          PRIMARY KEY,
   wes_bez      VARCHAR(80)     NOT NULL
);

CREATE TABLE wareneingangskontrolle(
  wek_nr                INTEGER         PRIMARY KEY,                                            --Nummer des Kontrollberichtes
  wek_w_wen             INTEGER         REFERENCES wendat       ON UPDATE CASCADE,              --Wareneingang der bemängelt wurde
  wek_ak_nr             VARCHAR(40)     REFERENCES art          ON UPDATE CASCADE,              --Fehlerhafter Artikel
  wek_l_krz             VARCHAR(21)     REFERENCES adk          ON UPDATE CASCADE,              --Kürzel des Lieferanten
  -- wek_awr_id(_obsolet)               INTEGER         REFERENCES auswlogrueck ON UPDATE CASCADE,              --Entfällt mit Umstellung Auswärtsbearbeitung (für Lief.schein Rücklieferung)
  wek_q_nr              INTEGER         REFERENCES qab          ON UPDATE CASCADE ON DELETE SET NULL,
  wek_def               BOOLEAN         NOT NULL DEFAULT FALSE,
  wek_dat               DATE,                                                           --Datum der Erfassung
  wek_txt               TEXT,                                                           --Zusatztext
  wek_txt_rtf           TEXT,
  wek_fehler            INTEGER         REFERENCES wekfehlerklasse,                     --Zugewiesene Fehlerart
  wek_pruefer           INTEGER         REFERENCES llv                                  --Mit.Nr. des Mitarbeiter der Prüfung bearbeitet
);


CREATE OR REPLACE FUNCTION wareneingangskontrolle__b_iu() RETURNS TRIGGER AS $$
  DECLARE lgid  INTEGER;
          offen BOOLEAN;
  BEGIN

    IF (tg_op = 'UPDATE') THEN
        --Definitivstatus geändert
        IF (new.wek_def <> old.wek_def) THEN

          --SYS.QAB-Admins, WEB-Admins und WEB-Pruefer dürfen WEK definitiv setzen
          IF new.wek_def THEN
            IF NOT user_in_group(current_user::varchar, 'SYS.QAB-Admin', 'SYS.WEB-Admin', 'SYS.WEB-Pruefer') THEN -- NOT current_user IN (SELECT usename FROM pg_group, pg_user WHERE usesysid=ANY(grolist) AND ((groname='SYS.QAB-Admin') OR (groname='SYS.WEB-Admin') OR (groname='SYS.WEB-Pruefer'))) THEN
              RAISE EXCEPTION 'current_user_is_not_QAB_or_WEB-Admin_or_WEB-Pruefer xtt10261';
            END IF;
          END IF;

          --SYS.QAB-Admins und WEB-Admins dürfen WEK definitiv setzen
          IF (NOT new.wek_def) THEN
            IF NOT user_in_group(current_user::varchar, 'SYS.QAB-Admin', 'SYS.WEB-Admin') THEN --  NOT current_user IN (SELECT usename FROM pg_group, pg_user WHERE usesysid=ANY(grolist) AND ((groname='SYS.QAB-Admin') OR (groname='SYS.WEB-Admin'))) THEN
              RAISE EXCEPTION 'current_user_is_not_QAB_or_WEB-Admin xtt10261';
            END IF;
          END IF;

          --Kontrollmerkmale Wareneingangsbericht prüfen vor definitiv setzen
          IF (new.wek_def) AND EXISTS(SELECT 1 FROM wareneingangskontrolle JOIN artpruefungtest ON aprt_weknr = wek_nr
                                                WHERE aprt_weknr = new.wek_nr AND NOT(aprt_passed OR aprt_failed))
          THEN
             RAISE EXCEPTION 'wareneingangskontrolle_NOT_aprt_passed_OR_aprt_failed__EXISTANT xtt10565';
          END IF;

          --Lagerort holen
          SELECT lg_id into lgid 
            FROM lag 
            JOIN wendat ON (lg_ort = w_lgort) AND (lg_chnr = w_lgchnr) AND (lg_aknr = w_aknr) 
           WHERE w_wen = new.wek_w_wen;

          --Wenn wir gerade definitiv gesetzt haben => Lagerort entsperren, Definitiv aufgehoben => Lagerort neu sperren
          IF (new.wek_def) THEN
            UPDATE lag SET lg_sperr = FALSE WHERE lg_id = lgid AND lg_sperr;
          ELSE
            UPDATE lag SET lg_sperr = TRUE WHERE lg_id = lgid AND NOT lg_sperr;
          END IF;

        END IF;
    END IF;

    --bei Neuanlage Nachricht an alle die eingetragen sind in Gruppe
    IF tg_op = 'INSERT' THEN
        PERFORM
         msg_SendMsg(CAST(current_user AS VARCHAR), 'Neuer Wareneingangsbericht '||new.wek_nr, '', CAST(usename AS VARCHAR))
        FROM
         pg_group,
         pg_user
        WHERE
         groname = 'SYS.QAB-Neu-Infomail'
        AND
         usesysid = ANY(grolist);
    END IF;

    RETURN new;
  END$$LANGUAGE plpgsql;


CREATE TRIGGER wareneingangskontrolle__b_iu
  BEFORE INSERT OR UPDATE
  ON wareneingangskontrolle
  FOR EACH ROW
  EXECUTE PROCEDURE wareneingangskontrolle__b_iu();


--Maßnahmen für einen bestimmten Wareneingangsbericht
CREATE TABLE wekmassnahmen
(
  wmas_id       SERIAL           PRIMARY KEY,
  wmas_wek_nr   INTEGER NOT NULL REFERENCES wareneingangskontrolle ON UPDATE CASCADE ON DELETE CASCADE,
  wmas_wes_id   INTEGER NOT NULL REFERENCES weksofort ON UPDATE CASCADE ON DELETE CASCADE
);

 --Jeder Maßnahme kann nur einmal pro Kontrolle erfolgen

CREATE UNIQUE INDEX wekmassnahmen_wmas_wek_nr_wmas_wes_id ON wekmassnahmen(wmas_wek_nr,wmas_wes_id);


CREATE OR REPLACE FUNCTION q_typ_to_str (IN qtyp INTEGER) RETURNS VARCHAR AS $$
   SELECT CASE $1
          WHEN 1 THEN lang_text(12320)
          WHEN 2 THEN lang_text(12321)
          WHEN 3 THEN lang_text(12322)
          WHEN 4 THEN lang_text(12323)
          WHEN 5 THEN lang_text(12324)
          WHEN 6 THEN lang_text(11994)
          WHEN 7 THEN lang_text(11993)
          ELSE $1::VARCHAR
    END
$$ LANGUAGE SQL;


--COC Herstellerprüfzertifikat http://redmine.prodat-sql.de/issues/4857
CREATE TABLE coc
(
  coc_id                        SERIAL NOT NULL PRIMARY KEY,      -- Eindeutiger Schlüssel
  coc_beld_dokunr               VARCHAR(30) NOT NULL,             -- Lieferschein-Nr/Dokumentnummer zum Beleg; Keine Cascade bisher, weil wenn der LFS veresehntl. gelöscht wird, auch der COC weg wäre
  coc_apint                     VARCHAR(10) REFERENCES llv(ll_db_usename) ON UPDATE CASCADE DEFAULT tsystem.current_user_ll_db_usename(), -- Ansprechpartner Intern...Standard angemeldeter Nutzer
  coc_definitiv                 BOOL NOT NULL DEFAULT FALSE,      -- Dokument ist definitiv
  coc_print                     BOOLEAN NOT NULL DEFAULT FALSE,   -- Druckstatus. Das Dokument wurde bereits ausgegeben.
  coc_ag_id                     INTEGER                           -- Optionales Feld, zur manuellen Verknüpfung bei freien Lieferscheinen
);

CREATE TABLE cocpos
(
  cocp_id                       SERIAL NOT NULL PRIMARY KEY,      -- Eindeutiger Schlüssel
  cocp_coc_id                   INTEGER NOT NULL REFERENCES coc ON UPDATE CASCADE ON DELETE CASCADE,
  cocp_belp_id                  INTEGER NOT NULL, --X TableContraints: REFERENCES belegpos      -- Lieferschein-Pos ID
  cocp_l_nr                     INTEGER, --Lagerabgangs-Nr
  cocp_sn                       TEXT,                                   -- Seriennummern
  cocp_txt                      TEXT,                                   -- Positionszusatztext bisher optional
  cocp_txt_rtf                  TEXT
);


-- #19744 QAB: Oberflächenkonfiguration
-- gibt die QAB-GUI-Konfigurationen nach ihrer Prio geordnet zurück
CREATE OR REPLACE FUNCTION qab_gui_config__get() RETURNS TABLE(
  qgc_id                        integer,        
  qgc_bez_xtt                   integer, 
  qgc_bez                       varchar,            
  qgc_pos                       integer,                         
  qgc_d2_visible                boolean,   
  qgc_d3_visible                boolean,   
  qgc_d4_visible                boolean,   
  qgc_d5_visible                boolean,   
  qgc_d6_visible                boolean,    
  qgc_d7_visible                boolean,   
  qgc_d8_visible                boolean,   
  qgc_bem                       text,
  qgc_standard                  boolean                             
) AS $$

  SELECT
    qgc_id,
    qgc_bez_xtt,
    prodat_languages.lang_text( qgc_bez_xtt ) AS qgc_bez,
    qgc_pos, 
    qgc_d2_visible, 
    qgc_d3_visible, 
    qgc_d4_visible, 
    qgc_d5_visible, 
    qgc_d6_visible, 
    qgc_d7_visible, 
    qgc_d8_visible,
    qgc_bem,
    qgc_standard
  FROM qab_gui_config   
  ORDER BY qgc_standard DESC, qgc_pos ASC;

$$ LANGUAGE sql STABLE;
--


-- keine leeren Statements am Ende vom Erstellen der DB erlaubt.
SELECT true;
